%---------------------------------------------------------------
%
% (i) Create comsol model for mechanics (geometry, material, mesh, physics)
% (ii) Create and solve the comsol model for optics
%
%---------------------------------------------------------------


function create_model(geom, Eval_mech)

    format long
    clear all;
    close all;

    tic;
    import com.comsol.model.*
    import com.comsol.model.util.*

    model = ModelUtil.create('Model');
    model.modelNode.create('comp1');
    geometry = model.geom.create('geom1',3);

    disp('Setting Paramters... ');
        
%%   parameters and definitions
    
    % Scale factors for CAD sweep
    geom.scale_all = 1.02;
    geom.scale_small_holes = 1.0702; % vary scale_small_holes

    % Parameters of the snowflake (slightly modified from Snowflake_topological_insulator paper)
    geom.mech.d = 220e-9;
    geom.mech.r = 180e-9*geom.scale_all;
    geom.mech.delta_r = 0; % Shift the Dirac cone in kx direction
    geom.mech.r1 = geom.mech.r + geom.mech.delta_r;
    geom.mech.r2 = (3*geom.mech.r - geom.mech.r1)/2;
    geom.mech.r3 = (3*geom.mech.r - geom.mech.r1)/2;
    geom.mech.w = 73e-9*geom.scale_all;
    geom.mech.a = 500e-9*geom.scale_all;
    geom.mech.rad_curv = 20e-9*geom.scale_all;
    geom.mech.rad_curv2 = 12e-9*geom.scale_all;
    geom.mech.resolution = 40; % Resolution of the snowflake polygon

    % Minimum distance between optical cavity hole and the snowflake hole
    geom.bridge_width = 70e-9*geom.scale_all;

    % Parameters of the optical cavity
    geom.opt.scale = 1.077*geom.scale_all; % Do not change this scaling parameter for CAD sweeps; Change scale_structure and scale_radius
    geom.opt.delta_a = -9e-9*geom.opt.scale;
    geom.opt.a1 = 425e-9*geom.opt.scale;
    geom.opt.a2 = geom.opt.a1+geom.opt.delta_a;
    geom.opt.a3 = geom.opt.a1+2*geom.opt.delta_a;
    geom.opt.a4 = geom.opt.a1+3*geom.opt.delta_a;
    geom.opt.r = 130e-9*geom.opt.scale*geom.scale_small_holes;
    geom.opt.W = 0;
    geom.opt.h = 130e-9*geom.opt.scale*geom.scale_small_holes;
    geom.opt.b = 150e-9*geom.opt.scale*geom.scale_small_holes;
    
    geom.opt.nh = 20;
    geom.opt.nv = 40;
    geom.opt.defect_holes = 2*geom.opt.nv;
    geom.opt.defect_width = geom.opt.r/geom.scale_small_holes*2+30e-9;
    geom.opt.defect_length = geom.opt.a1*100;
    geom.opt.cavity_rot = 0;
    geom.opt.cavity_shift = 4; % Cavity shift from the center of triangle
    geom.opt.scale_top_gap = 0.783333; %Scale factor of the radius in one of the triangle

    % Determining the scale factor of the snowflake so that optical cavity fits
    % inside the structure
    n = linspace(1,20,20);
    s = ((n+1)*geom.opt.a1 + 2*geom.opt.a2 + 2*geom.opt.a3 + geom.opt.a4)*sqrt(3)/geom.mech.a;
    index = 14; % 13 bad, 14 good
    num = n(index);
    geom.mech.scale = s(index);

    % Scaling the parameters of the snowflake by the above scale factor
    geom.mech.amech = geom.mech.a*geom.mech.scale;
    geom.mech.r = geom.mech.r*geom.mech.scale;
    geom.mech.r1 = geom.mech.r1*geom.mech.scale;
    geom.mech.r2 = geom.mech.r2*geom.mech.scale;
    geom.mech.r3 = geom.mech.r3*geom.mech.scale;
    geom.mech.w = geom.mech.w*geom.mech.scale;
    geom.mech.a = geom.mech.a*geom.mech.scale;
    geom.mech.rad_curv = geom.mech.rad_curv*geom.mech.scale;
    geom.mech.rad_curv2 = geom.mech.rad_curv2*geom.mech.scale;
    
    % Height of the snowflake triangle (not the triangle where topological edge
    % state is travelling)
    geom.mech.gap_defect_length = geom.mech.amech*sqrt(3)/2 - geom.mech.w*3/2;

    % Mesh parameters
    Eval_mech.hmax = 90e-9;
    Eval_mech.hmin = 25e-9;

    % Mirror symmetries
    Eval_mech.symm_z = 1; % about xy plane
    
    % .mph and .mat output files
    Eval_mech.filename = 'opt_sim';

    % Strip size
    geom.stack.num_domx = 1; % Number of unit cells in x-direction
    geom.stack.num_domy = 6; % Number of unit cells in y-direction = 2*num_domy+1
    geom.mech.fab_extra = 5.5e-6; % Extra length at the physical edge
    geom.mech.mirror_y = -geom.stack.num_domy*geom.mech.amech*sqrt(3)/2; % Position of mirror axis
    
    geom.mech.bdd_shift = geom.mech.amech/2;
    geom.mech.bdd_width = geom.mech.amech;
    geom.mech.bdd_length = 2*(geom.mech.amech/sqrt(3) - geom.mech.mirror_y + geom.mech.fab_extra);
    
    % Path along the k-space
    Eval_mech.res = 45;
    Eval_mech.kx = linspace(0,pi/(geom.mech.amech),Eval_mech.res);
    
    Eval_mech.NumberOfFrequencies = 25;
    Eval_mech.CentralFrequency = 348e6;
    Eval_mech.bands = zeros(Eval_mech.NumberOfFrequencies,size(Eval_mech.kx,2));
    
    % Optics parameters
    Eval_opt.hmax_air = 600e-9;
    
    Eval_opt.lambda = 1.5e-6; % Bounding box height = 2*lambda
    Eval_opt.qual_limit = 1e4; % Get the optical modes with Q greater than qual_limit
    
    Eval_opt.bc = 'TE'; % Mirror symmetry about xy plane
    
    Eval_opt.kx = 0; % Optical mode at gamma point
    Eval_opt.ky = 0;
    
    % Comsol output parameters
    Eval_opt.NumberOfFrequencies = 5;
    Eval_opt.CentralFrequency = 202.3857939791238e+12;
    Eval_opt.freq = zeros(Eval_opt.NumberOfFrequencies*2,1);
    Eval_opt.damp = zeros(Eval_opt.NumberOfFrequencies*2,1);
    Eval_opt.qual = zeros(Eval_opt.NumberOfFrequencies*2,1);
    Eval_opt.qual_limit = 1e4;
    
    % Constants
    c = 299792458;
%%   COMSOL-GLOBAL PARAMETERS

    model.param.set('d', num2str(geom.mech.d));
    model.param.set('r', num2str(geom.mech.r));
    model.param.set('w', num2str(geom.mech.w));
    model.param.set('a', num2str(geom.mech.a));
    model.param.set('a1', num2str(geom.opt.a1));
    model.param.set('a2', num2str(geom.opt.a2));
    model.param.set('a3', num2str(geom.opt.a3));
    model.param.set('a4', num2str(geom.opt.a4));
    model.param.set('r', num2str(geom.opt.r));
    model.param.set('W', num2str(geom.opt.W));
    model.param.set('height', num2str(geom.opt.h));
    model.param.set('b', num2str(geom.opt.b));
    model.param.set('scale_rad', num2str(geom.opt.scale_top_gap));
    model.param.set('lambda', num2str(Eval_opt.lambda));
    model.param.set('hmax', num2str(Eval_mech.hmax));
    model.param.set('hmin', num2str(Eval_mech.hmin));
    model.param.set('hmax_air', num2str(Eval_opt.hmax_air));

    mphsave(model, Eval_mech.filename);

%%   COMSOL-GEOMETRY
    
    disp('Building Geometry... ');
    
    workplane = geometry.feature.create('wp1','WorkPlane');
    
    model = draw_snowflakes_hexagon(geom, model, workplane);
    
    model = draw_cavity_hexagon(geom, model, workplane);
     
    model = draw_cavity_hexagon_domain_wall(geom, model, workplane);
    
    %Bounding surface of the geometry
    pol_x = geom.mech.amech/2 + geom.mech.bdd_shift + [-geom.mech.bdd_width/2 -geom.mech.bdd_width/2 geom.mech.bdd_width/2 geom.mech.bdd_width/2 -geom.mech.bdd_width/2];
    pol_y = geom.mech.mirror_y + [-geom.mech.bdd_length/2 geom.mech.bdd_length/2 geom.mech.bdd_length/2 -geom.mech.bdd_length/2 -geom.mech.bdd_length/2];
    
    mech = workplane.geom.create('bound_pol', 'Polygon');
    mech.set('source', 'table');
    for pol_i=1:length(pol_x(1:end-1))
        mech.setIndex('table', num2str(pol_x(pol_i),'%10.9e'), pol_i-1, 0);  %value,row, coloumn
        mech.setIndex('table', num2str(pol_y(pol_i),'%10.9e'), pol_i-1, 1);
    end 
    
    workplane.geom.create('uni7', 'Union');
    workplane.geom.feature('uni7').selection('input').set({'arr1' 'arr2' 'uni_all' 'mir1_domain_wall' 'uni_all_domain_wall'});
    
%     workplane.geom.create('uni7', 'Union');
%     workplane.geom.feature('uni7').selection('input').set({'arr1' 'arr2'});
    
    copy = workplane.geom.create('disp', 'Copy');
    copy.selection('input').set('uni7');
    copy.set('displx', num2str(geom.mech.amech,'%10.9e'));
    
    workplane.geom.create('dif1', 'Difference');
    workplane.geom.feature('dif1').selection('input').set({'bound_pol'});
    workplane.geom.feature('dif1').selection('input2').set({'uni7' 'disp'});
 
    ext = geometry.feature.create('ext1', 'Extrude');
    ext.set('workplane', 'wp1');
    ext.selection('input').set({'wp1'});
    ext.setIndex('distance', 'd/2', 0);
    
    workplane = geometry.feature.create('wp2','WorkPlane');
    workplane.set('unite', true);
    workplane.set('quickplane', 'xz');
    workplane.set('quicky', geom.mech.mirror_y - geom.mech.amech*sqrt(3)/2);
    
    workplane = geometry.feature.create('wp3','WorkPlane');
    workplane.set('unite', true);
    workplane.set('quickplane', 'xz');
    workplane.set('quicky', geom.mech.mirror_y + geom.mech.amech*sqrt(3)/2);
    
    part_dom = geometry.create('pard1', 'PartitionDomains');
    part_dom.selection('domain').set('ext1', 1);
    part_dom.set('workplane', 'wp2');
    
    part_dom = geometry.create('pard2', 'PartitionDomains');
    part_dom.selection('domain').set('pard1', 2);
    part_dom.set('workplane', 'wp3');
    
    % Air column
    workplane = geometry.feature.create('wp4','WorkPlane');
    
    pol_air_x = geom.mech.amech + [-geom.mech.bdd_width/2 ...
        -geom.mech.bdd_width/2 geom.mech.bdd_width/2 ...
        geom.mech.bdd_width/2 -geom.mech.bdd_width/2];
    pol_air_y = geom.mech.mirror_y + [-geom.mech.amech*sqrt(3)/2 ...
        geom.mech.amech*sqrt(3)/2 geom.mech.amech*sqrt(3)/2 ...
        -geom.mech.amech*sqrt(3)/2 -geom.mech.amech*sqrt(3)/2];
    
    mech = workplane.geom.create('unit_cell_bdd_air', 'Polygon');
    mech.set('source', 'table');
    for i=1:length(pol_air_x(1:end-1))
        mech.setIndex('table', num2str(pol_air_x(i),'%10.9e'), i-1, 0);  %value,row, coloumn
        mech.setIndex('table', num2str(pol_air_y(i),'%10.9e'), i-1, 1);
    end
    
    extair = geometry.feature.create('ext2', 'Extrude');
    extair.set('workplane', 'wp4');
    extair.selection('input').set({'wp4'});
    extair.setIndex('distance', '2*lambda', 0);
    
    geometry.run;
    figure(1)
    mphgeom(model);
    view(0,90);
    mphsave(model,Eval_mech.filename)
    
%%   COMSOL-MATERIAL
    
    disp('Setting Material...');
    
    Silicon = model.material.create('mat2', 'Common', 'comp1');
    Silicon.name('Silicon');
    Silicon.propertyGroup('def').set('relpermeability', '1');
    Silicon.propertyGroup('def').set('electricconductivity', '1e-12[S/m]');
    Silicon.propertyGroup('def').set('relpermittivity', '11.7');

    model.material('mat2').label('Si - Silicon (single-crystal, anisotropic) 1');
    model.material('mat2').set('family', 'custom');
    model.material('mat2').set('lighting', 'cooktorrance');
    model.material('mat2').set('specular', 'custom');
    model.material('mat2').set('customspecular', [0.7843137254901961 1 1]);
    model.material('mat2').set('fresnel', 0.9);
    model.material('mat2').set('roughness', 0.1);
    model.material('mat2').set('shininess', 200);
    model.material('mat2').set('diffuse', 'custom');
    model.material('mat2').set('customdiffuse', [0.6666666666666666 0.6666666666666666 0.7058823529411765]);
    model.material('mat2').set('ambient', 'custom');
    model.material('mat2').set('customambient', [0.6666666666666666 0.6666666666666666 0.7058823529411765]);
    model.material('mat2').set('fresnel', 0.7);
    model.material('mat2').set('roughness', 0.5);
    model.material('mat2').propertyGroup('def').set('density', '2330[kg/m^3]');
    model.material('mat2').propertyGroup.create('Anisotropic', 'Anisotropic');    
    model.material('mat2').propertyGroup('Anisotropic').set('D', {'166[GPa]' '64[GPa]' '166[GPa]' '64[GPa]' '64[GPa]' '166[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '80[GPa]'  ...
    '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '80[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]'  ...
    '80[GPa]'});
    
    epsilon = 3e-9;
    coordBox = [-1, 1; -1, 1; -epsilon, geom.mech.d/2+epsilon];
    si_domain = mphselectbox(model,'geom1',coordBox,'domain');
    Silicon.selection.set(si_domain);
    
    Air = model.material.create('Air');
    Air.name('Air');
    Air.propertyGroup('def').set('relpermeability', '1');
    Air.propertyGroup('def').set('relpermittivity', '1');
    Air.propertyGroup('def').set('electricconductivity', '0[S/m]');
    
    coordBox = [-1, 1; -1, 1; -1, 1];
    all_domain = mphselectbox(model,'geom1',coordBox,'domain');
    air_domain = setdiff(all_domain, si_domain);
    Air.selection.set(air_domain);
   
    mphsave(model, Eval_mech.filename);

%%   COMSOL-MESH
    
    disp('Meshing...');
    
    mesh = model.mesh.create('mesh1', 'geom1');
    
    coordBox = [-1, 1; -1, 1; -epsilon, epsilon];
    sbz_all = mphselectbox(model,'geom1',coordBox,'boundary');
    sbz_air = mphselectbox(model,'geom1',coordBox,'boundary',...
        'adjnumber',air_domain);
    
    coordBox = [geom.mech.amech-geom.mech.bdd_width/2 - ...
        epsilon, geom.mech.amech-geom.mech.bdd_width/2 + epsilon; ...
        -1, 1; -epsilon, geom.mech.d/2 + epsilon];
    l1 = mphselectbox(model,'geom1',coordBox,'boundary');
    l1_all_edge = mphselectbox(model,'geom1',coordBox,'edge');
    
    coordBox = [-epsilon, epsilon; -1, 1; -epsilon, epsilon];
    l1_edge_bot = mphselectbox(model,'geom1',coordBox,'edge');
    
    coordBox = [-epsilon, epsilon; -1, 1; geom.mech.d/2-epsilon, geom.mech.d/2+epsilon];
    l1_edge_top = mphselectbox(model,'geom1',coordBox,'edge');
    
    l1_edge = setdiff(l1_all_edge, horzcat(l1_edge_bot, l1_edge_top));
    
    coordBox = [geom.mech.amech+geom.mech.bdd_width/2 - ...
        epsilon, geom.mech.amech+geom.mech.bdd_width/2 + epsilon; ...
        -1, 1; -epsilon, geom.mech.d/2 + epsilon];
    l2 = mphselectbox(model,'geom1',coordBox,'boundary');
    
%     freequad = mesh.create('fq1', 'FreeQuad');
%     freequad.selection.set([l1]);
%     freequad.create('dis1', 'Distribution');
%     freequad.feature('dis1').selection.set([l1_edge]);
%     freequad.feature('dis1').set('numelem', 2);

    freequad = mesh.create('fq1', 'FreeQuad');
    freequad.selection.set([l1]);
    freequad.create('size1', 'Size');
    freequad.feature('size1').set('custom', true);
    freequad.feature('size1').set('hmaxactive', true);
    freequad.feature('size1').set('hmax', 'hmax');
    freequad.feature('size1').set('hminactive', true);
    freequad.feature('size1').set('hmin', 'hmin');
    
    copy_face = mesh.create('cpf1', 'CopyFace');
    copy_face.selection('source').set([l1]);
    copy_face.selection('destination').set([l2]);

    mesh.create('ftri1', 'FreeTri');
    mesh.feature('ftri1').create('size1', 'Size');
    mesh.feature('ftri1').feature('size1').set('custom', 'on');
    mesh.feature('ftri1').feature('size1').set('hmax', 'hmax');
    mesh.feature('ftri1').feature('size1').set('hmin', 'hmin');
    mesh.feature('ftri1').selection.set([setdiff(sbz_all, sbz_air)]);
    
    mesh.create('swe1', 'Sweep');
    mesh.feature('swe1').create('dis1', 'Distribution');
    mesh.feature('swe1').feature('dis1').set('numelem', num2str(2)); %Number of mesh layers = 2
    mesh.feature('swe1').selection.set(si_domain);
    mesh.feature('swe1').feature('dis1').selection.set(si_domain);
    
    mesh.create('ftet2', 'FreeTet');
    mesh.feature('ftet2').create('size1', 'Size');
    mesh.feature('ftet2').feature('size1').set('custom', 'on');
    mesh.feature('ftet2').feature('size1').set('hmax', 'hmax_air');
    mesh.feature('ftet2').feature('size1').set('hmin', 'hmin');
    mesh.feature('ftet2').selection.set(air_domain);
    mesh.feature('ftet2').feature('size1').selection.set(air_domain);
    
    mesh.run;
    figure(2)
    mphmesh(model)
    mphsave(model, Eval_mech.filename);

%%   COMSOL-PHYSICS
    
    disp('Setting Physics...');
    
    physics_opt = model.physics.create('emw', 'ElectromagneticWaves', 'geom1');
    
    % Selecting domains
    coordBox = [-1, 1; geom.mech.mirror_y - ...
        geom.mech.amech*sqrt(3)/2 - epsilon, ...
        geom.mech.mirror_y + geom.mech.amech*sqrt(3)/2 + epsilon; ...
        -1, 1];
    dom_opt = mphselectbox(model,'geom1',coordBox,'domain');
    
    model.physics('emw').selection.set(dom_opt);
    
    % Mirror symmetry
    coordBox = [-1, 1; geom.mech.mirror_y - ...
        geom.mech.amech*sqrt(3)/2 - epsilon, ...
        geom.mech.mirror_y + geom.mech.amech*sqrt(3)/2 + epsilon; ...
        -epsilon, epsilon];
    sbz_opt = mphselectbox(model,'geom1',coordBox,'boundary');
    
    if (Eval_opt.bc == 'TE')
        model.physics('emw').create('pmc1', 'PerfectMagneticConductor', 2);
        model.physics('emw').feature('pmc1').selection.set(sbz_opt);
    elseif(Eval_opt.bc == 'TM')
        model.physics('emw').create('pec2', 'PerfectElectricConductor', 2);
        model.physics('emw').feature('pec2').selection.set(sbz_opt);
    end
    
    % Scattering boundary condition
    coordBox = [-1, 1; -1, 1; 2*Eval_opt.lambda-epsilon, ...
        2*Eval_opt.lambda+epsilon];
    sc1t = mphselectbox(model,'geom1',coordBox,'boundary');
    model.physics('emw').feature.create('sctr1', 'Scattering', 2);    % What does 2 stand for here?
    model.physics('emw').feature('sctr1').selection.set([sc1t]);
    
    % Periodic boundary condition
    coordBox = [geom.mech.amech-geom.mech.bdd_width/2 - ...
        epsilon, geom.mech.amech-geom.mech.bdd_width/2 + epsilon; ...
        geom.mech.mirror_y - geom.mech.amech*sqrt(3)/2 - epsilon, ...
        geom.mech.mirror_y + geom.mech.amech*sqrt(3)/2 + epsilon; ...
        -1, 1];
    l1_opt = mphselectbox(model,'geom1',coordBox,'boundary');
    
    coordBox = [geom.mech.amech+geom.mech.bdd_width/2 - ...
        epsilon, geom.mech.amech+geom.mech.bdd_width/2 + epsilon; ...
        geom.mech.mirror_y - geom.mech.amech*sqrt(3)/2 - epsilon, ...
        geom.mech.mirror_y + geom.mech.amech*sqrt(3)/2 + epsilon; ...
        -1, 1];
    l2_opt = mphselectbox(model,'geom1',coordBox,'boundary');
    
    physics_opt.feature.create('pc1', 'PeriodicCondition', 2);    % What does 2 stand for here?
    physics_opt.feature('pc1').set('PeriodicType', 'Floquet');
    physics_opt.feature('pc1').selection.set([l1_opt l2_opt]);
    
    % Mechanics
    physics_mech = model.physics.create('solid', 'SolidMechanics', 'geom1');
    physics_mech.feature('lemm1').set('SolidModel', 'Anisotropic');
    
    % Selecting domains
    coordBox = [-1, 1; -1, 1; -epsilon, geom.mech.d/2 + epsilon];
    dom_mech = mphselectbox(model,'geom1',coordBox,'domain');
    
    model.physics('solid').selection.set(dom_mech);
    
    coordBox = geom.mech.mirror_y + [-1, 1; -geom.mech.bdd_length/2 - epsilon, -geom.mech.bdd_length/2 + epsilon; -1, 1];
    fixed1 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    coordBox = geom.mech.mirror_y + [-1, 1; geom.mech.bdd_length/2 - epsilon, geom.mech.bdd_length/2 + epsilon; -1, 1];
    fixed2 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    physics_mech.feature.create('pc1', 'PeriodicCondition', 2);    % What does 2 stand for here?
    physics_mech.feature('pc1').set('PeriodicType', 'Floquet');
    physics_mech.feature('pc1').selection.set([l1 l2]);
    
    if Eval_mech.symm_z == +1
        physics_mech.feature.create('sym1', 'SymmetrySolid', 2);
        physics_mech.feature('sym1').selection.set([sbz_all]);
    elseif Eval_mech.symm_z == -1
        physics_mech.feature.create('as1', 'Antisymmetry', 2);
        physics_mech.feature('as1').selection.set([sbz_all]);
    end
    
    model.physics('solid').create('fix1', 'Fixed', 2);
    model.physics('solid').feature('fix1').selection.set([fixed1 fixed2]);
  
    mphsave(model, Eval_mech.filename);

%%  COMSOL OPTICS STUDY
  
    disp('Launching optics study... ');
    
    study = model.study.create('std');
    studyEf = study.feature.create('eig', 'Eigenfrequency');
    studyEf.set('eigunit', 'Hz');
    studyEf.activate('emw', true);
    studyEf.activate('solid', false);
    studyEf.set('neigs', num2str(Eval_opt.NumberOfFrequencies));
    studyEf.set('shift', num2str(Eval_opt.CentralFrequency));
    
    model.physics('emw').feature('pc1').set('kFloquet', ...
        {num2str(Eval_opt.kx,'%10.9e') num2str(Eval_opt.ky,'%10.9e') '0'});

    mphsave(model, strcat(Eval_mech.filename, '_setup'));
    
    study.run;
    
    mphsave(model, Eval_mech.filename);
    
    data_freq = mpheval(model,'real(freq)','selection',1,'edim',...
        'point','dataonly','on');
    data_damp = mpheval(model,'imag(freq)','selection',1,'edim',...
        'point','dataonly','on');
    
    for j=1:min(size(Eval_opt.freq,1), size(data_freq,1))
        Eval_opt.freq(j)=data_freq(j);
        Eval_opt.damp(j)=data_damp(j);
        
        Eval_opt.qual(j) = Eval_opt.freq(j)/(2*Eval_opt.damp(j));
    end
    
    [Eval_opt.quality, Eval_opt.index] = max(Eval_opt.qual);
    Eval_opt.frequency = Eval_opt.freq(Eval_opt.index);
    Eval_opt.damping = Eval_opt.damp(Eval_opt.index);
    
    save(Eval_mech.filename, 'Eval_mech', 'geom');
    
    %% Plotting all the optical modes with Q>1e4

    index = find(real(Eval_opt.qual)>Eval_opt.qual_limit);

    model.result.create('pg2', 'PlotGroup3D');
    model.result('pg2').set('frametype', 'spatial');
    model.result('pg2').set('data', 'dset1');
    model.result('pg2').feature.create('mslc1', 'Multislice');
    model.result('pg2').feature('mslc1').label('Multislice');
    model.result('pg2').feature('mslc1').set('data', 'parent');
    model.result('pg2').feature('mslc1').set('expr', 'emw.Ex');
    model.result('pg2').feature('mslc1').set('descr', 'Electric field, x component');

    model.result('pg2').feature('mslc1').set('xnumber', '0');
    model.result('pg2').feature('mslc1').set('ynumber', '0');
    model.result('pg2').feature('mslc1').set('multiplanezmethod', 'coord');
    model.result('pg2').feature('mslc1').set('zcoord', '0');

    for j=1:length(index)
        temp = figure;
        clf;
        hold on;

        model.result('pg2').set('looplevel', {num2str(index(j))});
        mphplot(model, 'pg2');
        view(0,90)
        box on

        savefig(temp, strcat('index_', num2str(index(j)), ...
            '_wavelength_', num2str(c/Eval_opt.freq(index(j))), ...
            '_Qt_ ', num2str(Eval_opt.qual(index(j))), '.fig'))
    end
    
%%    
    Eval_mech.time = toc;
    
    save(Eval_mech.filename, 'Eval_mech', 'Eval_opt', 'geom');
%     
end
